#version 130
#extension GL_EXT_gpu_shader4 : enable
// the version and open GL extension
// should be the first line of the shader
/////////////////////////////////////////////////////////////////////////////////
// random cubic Truchet flowMod01.fsh  by   mattz 
//https://www.shadertoy.com/view/MtSyRz
//Licence : Creative Commons Attribution-ShareAlike 4.0
//http://creativecommons.org/licences/by-sa/4.0
// Adapted, trivialy, for use in VGHD player
/////////////////////////////////////////////
uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

#define iTime u_Elapsed*0.314159  //*0.1666
#define iResolution u_WindowSize

//#define mouse AUTO_MOUSE
//#define MOUSE_SPEED vec2(vec2(0.5,0.577777) * 0.25)
//#define MOUSE_POS   vec2((1.0+cos(iTime*MOUSE_SPEED))*u_WindowSize/2.0)
//#define MOUSE_PRESS vec2(0.0,0.0)
//#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )
//#define RIGID_SCROLL
// alternatively use static mouse definition
#define iMouse vec4(0.0,0.0, 0.0,0.0)
//#define iMouse vec4(512,256,180,120)
uniform sampler2D texture0;
uniform sampler2D texture1;
uniform sampler2D texture2;
uniform sampler2D texture3;
vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}
#define texture2D texture2D_Fract

/* 
    random cubic Truchet flow, by mattz
    License Creative Commons Attribution-NonCommercial-ShareAlike 3.0 Unported License.

	Inspired by:

      Truchet Marbles (Simple) by shau - https://www.shadertoy.com/view/MtlcDs
      Square Truchet Flow by Shane - https://www.shadertoy.com/view/XtfyDX


    Keys:

      D - toggle direction arrows (rotations/reflections of canonical cell)
      M - toggle camera motion
      S - toggle spheres
      T - toggle Truchet torii
      X - toggle X-axis alternation


    Mouse to bottom-left to get default viewing angle.


    Analogous to Shane's 2D shader, this fixes the flow directions along cube faces
    in a repeating fashion. I randomize the torus orientations while confirming to the
    predetermined flow directions.

    See the documentation of the truchet function below for the meat.

    This calls atan to do the shading along each torus segment, but I make sure
    to only call it once per pixel when actually shading. 

*/

// Bunch of globals/constants:
const int rayiter = 60;
const float dmax = 20.0;
vec3 L = normalize(vec3(-0.7, 1.0, -1.0));
mat3 Rview;

float move_camera = 1.0;
float show_spheres = 1.0;
float show_directions = 0.0;
float show_truchet = 1.0;
float alternate_x = 1.0;

const float HALFPI = 1.5707963267948966;
const float TUBE_SIZE = 0.015;
const float SPHERE_SIZE = 0.06;

const float ARROW_RAD = 0.025;
vec2 ARROW_HEAD_SLOPE = normalize(vec2(1, 2));
    
const float ARROW_BODY_LENGTH = 0.3;
const float ARROW_HEAD_LENGTH = 0.1;

/* RGB from hue. */
vec3 hue(float h) {
    vec3 c = mod(h*6.0 + vec3(2, 0, 4), 6.0);
    return h >= 1.0 ? vec3(h-1.0) : clamp(min(c, -c+4.0), 0.0, 1.0);
}

/* Rotate about x-axis */
mat3 rotX(in float t) {
    float cx = cos(t), sx = sin(t);
    return mat3(1., 0, 0, 
                0, cx, sx,
                0, -sx, cx);
}

/* Rotate about y-axis */
mat3 rotY(in float t) {
    float cy = cos(t), sy = sin(t);
    return mat3(cy, 0, -sy,
                0, 1., 0,
                sy, 0, cy);

}

/* From https://www.shadertoy.com/view/4djSRW */
#define HASHSCALE1 .1031

float hash13(vec3 p3) {
	p3  = fract(p3 * HASHSCALE1);
    p3 += dot(p3, p3.yzx + 19.19);
    return fract((p3.x + p3.y) * p3.z);
}

/* Distance to 3D arrow - steal this if you like! */
float sdArrow(vec3 p, vec3 d) {
    
    // component along
    float t = dot(p, d);
    
    // component perp
    float n = length(p - t*d);
    
	// distance to cylinder body
    float dist = n - ARROW_RAD;
    
    // account for arrowhead size
    t += 0.5*ARROW_HEAD_LENGTH;
    
    // body caps
    dist = max(dist, abs(t)-0.5*ARROW_BODY_LENGTH);
    
    // move to end of arrow
    t -= 0.5*ARROW_BODY_LENGTH;
    
    // cone head
    dist = min(dist, max(-t, dot(ARROW_HEAD_SLOPE, vec2(t-ARROW_HEAD_LENGTH, n))));
    
    return dist;
    
}


/* Check distance to torus at origin, perpendicular to z axis. */
float sdTorus(vec3 u) {
    
    vec2 q = vec2(length(u.xy)-0.5,u.z);
    float d = length(q) - TUBE_SIZE;
    d = max(d, max(u.x, u.y));

    return d;
    
}

/* Distance to ball along torus segment orbiting origin (same frame as sdTorus). */
float sdBall(vec3 u) {
    
    float theta = HALFPI * fract(0.2*iTime);

	// Is this cheaper than separate sin & cos? I dunno.    
    vec2 sc = 0.5*sin(vec2(theta, theta+HALFPI));
    
    // Need multiple checks to handle ball crossing cube face boundaries!
    float d = length(u + vec3(sc.x, sc.y, 0));
    d = min(d, length(u + vec3(sc.y, -sc.x, 0)));
    d = min(d, length(u + vec3(-sc.y, sc.x, 0)));
    
    return d - SPHERE_SIZE;

}

/* Compute edge-relative coords and torus distance given:

    - p is point in cube with vertices at +/- 0.5:
    - src is outward facing normal of flow source face 
    - dst is outward facing normal of flow sink face

   Returns a vec4 (ux uy uz d) where:

     - ux is distance from src face (negative inside cube)
     - uy is distance from dst face (also negative inside cube)
     - uz is projected distance along edge from center (positive or negative)
     - d is distance to torus.

*/
vec4 checkTorus(vec3 p, vec3 src, vec3 dst) {
    
    vec3 n = cross(dst, src);
    vec3 ctr = 0.5*(src+dst);
    
    p -= ctr;
    
    vec3 u = p * mat3(src, dst, n);
    
    return vec4(u, sdTorus(u));
    
}

/* Chooses vec4 with least last element (useful for comparing checkTorus output) */
vec4 tmin(vec4 a, vec4 b) {
    
    return a.w < b.w ? a : b;
    
}


/* This is the workhorse of the Truchet arrangement. Input argument is
   an arbitrary 3D point, which is modified to become relative to cell
   center and orientation upon output. Also outputs a mat3 of the format
   output by sdTorus above. 

   As you can see by enabling cell directions, every cubic cell in the
   Truchet tiling has three flow inputs and three outputs. The "canonical 
   cell" has flow inputs on the +X, -Y, and +Y faces (and flow outputs on
   the -X, -Z, +Z) faces.

   In order to get these to tile space, we need to swap Y with Z in a
   checkerboard pattern along the YZ plane.

   Also, it looks obviously "fake" (in my opinion, at least) to have all 
   of the flow from +X to -X, so I also alternate the X direction on
   successive Y levels. 

   So, we now have the "canonical cell" potentially with Y/Z swapped, and/or
   X negated. There are four possible torus segment arrangements within 
   the canonical cell. The table below shows for each input face (+X, -Y, +Y),
   what output face it connects to:

     case | +X -Y +Y
     -----+----------
        0 | -Z -X +Z
        1 | -Z +Z -X
        2 | +Z -X -Z
        3 | +Z -Z -X
       
   We choose one of these cases at random for each cell, and get the torus 
   distance for the given point.

*/
vec4 truchet(inout vec3 pos) {

    // Shift by 0.5 to place world origin at vertex, not cell center
    pos -= 0.5;
    
    // Find center of nearest cell
    vec3 ctr = floor(pos+0.5);
    
    // Subtract off difference
    pos -= ctr;
    
    // Alternating sign on each axis
    vec3 s = sign(mod(ctr, 2.0) - 0.5);
    
    // Swap Y and Z in checkerboard pattern.
    if (s.y * s.z > 0.0) { pos.yz = pos.zy; }

    // Alternate sign on X within cell if desired
    if (alternate_x > 0.0) { pos.x *= -s.y; }

    // Get case and set up destination axes
    float k = hash13(ctr) * 4.0;
    
    mat3 dst = mat3(0, 0, -1, 
                    -1, 0, 0, 
                    0, 0, 1);                
    
    if (k < 2.0) {
        if (k < 1.0) {
            // NOP - just use setup above
            // dst = dst
        } else {
            dst = mat3(dst[0], dst[2], dst[1]); 
        }
    } else {
        if (k < 3.0) {
            dst = mat3(dst[2], dst[0], dst[1]); 
        } else {
            dst = mat3(dst[2], dst[1], dst[0]); 
        }
    }
    
    // Handle +X face
    vec4 t = checkTorus(pos, vec3(1, 0, 0), dst[0]);
    
    // Handle +Y face
    t = tmin(t, checkTorus(pos, vec3(0, 1, 0), dst[1]));
    
    // Handle -Y face
    t = tmin(t, checkTorus(pos, vec3(0, -1, 0), dst[2]));
    
    return t;

    
}

/* Boolean union of solids for map function below */
vec2 dmin(vec2 a, vec2 b) {
    return a.x < b.x ? a : b;
}

/* Distance function to scene */
vec2 map(in vec3 pos) {	
    
    vec4 t = truchet(pos);
    
    vec2 dm = vec2(1e5, -1);
    
    if (show_truchet != 0.0) {
        dm = dmin(dm, vec2(t.w, 2.0));
    }
    
    if (show_spheres != 0.0) {
        dm = dmin(dm, vec2(sdBall(t.xyz), 1.1));
    }

    if (show_directions != 0.0) {
        dm = dmin(dm, vec2(sdArrow(pos - vec3(0.5, 0, 0), vec3(-1, 0, 0)), 0));
        dm = dmin(dm, vec2(sdArrow(pos + vec3(0.5, 0, 0), vec3(-1, 0, 0)), 0));
        dm = dmin(dm, vec2(sdArrow(pos - vec3(0, 0.5, 0), vec3(0, -1, 0)), 0.3333));
        dm = dmin(dm, vec2(sdArrow(pos + vec3(0, 0.5, 0), vec3(0, 1, 0)), 0.3333));
        dm = dmin(dm, vec2(sdArrow(pos - vec3(0, 0, 0.5), vec3(0, 0, 1)), 0.6666));
        dm = dmin(dm, vec2(sdArrow(pos + vec3(0, 0, 0.5), vec3(0, 0, -1)), 0.6666));
    }

    return dm;

}


/* IQ's normal calculation. */
vec3 calcNormal( in vec3 pos ) {
    vec3 eps = vec3( 0.001, 0.0, 0.0 );
    vec3 nor = vec3(
        map(pos+eps.xyy).x - map(pos-eps.xyy).x,
        map(pos+eps.yxy).x - map(pos-eps.yxy).x,
        map(pos+eps.yyx).x - map(pos-eps.yyx).x );
    return normalize(nor);
}


/* IQ's distance marcher. */
vec2 castRay( in vec3 ro, in vec3 rd, in float maxd ) {

    const float precis = 0.002;   
    float h=2.0*precis;

    float t = 0.0;
    float m = -1.0;

    for( int i=0; i<rayiter; i++ )
    {
        if( abs(h)<precis||t>maxd ) continue;//break;
        t += h;
        vec2 res = map( ro+rd*t );
        h = res.x;
        m = res.y;        
    }    

    if (t > maxd) {
        m = -1.0;
    }

    return vec2(t, m);

}

/* Only clever thing about the shading is to postpone the expensive atan call
   until the last possible second. It should buy us a bit of framerate on 
   slower cards, but I was getting 60FPS before I decided to do this. Meh.
 */
vec3 shade( in vec3 ro, in vec3 rd ){

    vec2 tm = castRay(ro, rd, dmax);        

    vec3 c;

    if (tm.y < 0.0) { 

        // miss
        return vec3(1.0);

    } else {        

        // hit
        vec3 pos = ro + tm.x*rd;
        vec3 n = calcNormal(pos);

        
        if (tm.y >= 2.0) { 
            
            // material 2 means we hit a torus, so use atan-based rainbow map
            vec4 t = truchet(pos);
            tm.y = fract( atan(t.y, t.x)/HALFPI - 0.25*iTime );
            
        }

        vec3 color = hue(tm.y);

        vec3 diffamb = (0.5*clamp(dot(n,L), 0.0, 1.0)+0.5) * color;
        vec3 R = 2.0*n*dot(n,L)-L;
        float spec = 0.3*pow(clamp(-dot(R, rd), 0.0, 1.0), 20.0);
        vec3 c = diffamb + spec;
        
        return mix(c, vec3(1), 1.0-exp(-0.25*tm.x));

    }


}

/* Compare key state to default state. */
float keyState(float key, float default_state) {
    return abs( texture2D(texture0, vec2(key, 0.75)).x - default_state );
}

const float KEY_D = 68.5/256.0;
const float KEY_M = 77.5/256.0;
const float KEY_S = 83.5/256.0;
const float KEY_T = 84.5/256.0;
const float KEY_X = 88.5/256.0;

/* Finally, our main function: */

//void mainImage( out vec4 fragColor, in vec2 fragCoord )
///////////////////////////////////////////////////////////////////////////////// 
// need to convert this from a void to a function and call it by adding
// a void main(void) { to the end of the shader
// what type of variable will the function return?, it is a color and needs to be a vec4
// change void to vec4 
//void MainImage(out vec4 fragColor, in vec2 fragCoord) 
vec4 mainImage( out vec4 fragColor, in vec2 fragCoord )
{ 
    
    show_directions = keyState(KEY_D, show_directions);
    move_camera = keyState(KEY_M, move_camera);
    show_truchet = keyState(KEY_T, show_truchet);
    show_spheres = keyState(KEY_S, show_spheres);
    alternate_x = keyState(KEY_X, alternate_x);

    vec2 uv = (fragCoord.xy - .5*iResolution.xy) * 0.8 / (iResolution.y);
    
    const vec3 tgt = vec3(0.0, 0.0, 0.0);
    const vec3 cpos = vec3(0, 0, 5);
    const vec3 up = vec3(0, 1, 0);

    vec3 rz = normalize(tgt - cpos),
        rx = normalize(cross(rz,vec3(0,1.,0))),
        ry = cross(rx,rz);


    vec2 mpos;

    if (max(iMouse.x, iMouse.y) > 20.0) { 
        mpos.xy = iMouse.xy;
    } else {
        mpos = iResolution.xy * vec2(0.432, 0.415);
    }
    
    float thetax = (mpos.y - .5*iResolution.y) * -4.0*HALFPI/iResolution.y; 
    float thetay = (mpos.x - .5*iResolution.x) * 8.0*HALFPI/iResolution.x; 


    Rview = mat3(rx,ry,rz)*rotY(thetay)*rotX(thetax); 
    L = Rview*L;

   	/* Render. */
    vec3 rd = Rview*normalize(vec3(uv, 1.)),
        ro = tgt + Rview*vec3(0,0,-length(cpos-tgt));
    
    if (move_camera != 0.0) {
        ro -= 0.15*iTime;
    }

    fragColor.xyz = shade(ro, rd);
/////////////////////////////////////////////////////////////////////////////////
//the function needs to return a value. 
//it needs to be a vec4
//we will return the varable fragColor 
// usual place for fragColor = vec4( color, 1.0 ); bring the } down below
return fragColor; 
}

///////////////////////////////////////////////////////////////////////////////// 
void main(void) { // this will be run for every pixel of gl_FragCoord.xy
vec4 vTexCoord = gl_TexCoord[0];
vec4 fragColor = vec4(1.0); // initialize variable fragColor as a vec4 
vec4 cc = mainImage(fragColor, gl_FragCoord.xy); // call function mainImage and assign the return vec4 to cc
gl_FragColor = vec4(cc) * gl_Color; // set the pixel to the value of vec4 cc  and..
gl_FragColor.a = length(gl_FragColor.rgb);
}

// ..uses the values of any Color: or Opacity:
// clauses (and any Animate clauses applied to these properties) 
// appearing in the Sprite, Quad or other node invoking the shader 
// in the .scn file.

